//-----< FS4.C >----------------------------------------------------
//
// Drake 4 Line Receiver Crystal Synthesizer
//
// By: S.C.Hageman - 23May02
//
// www.AnalogHome.com
//
//-------------------------------------------------------------------
// Copyright By: Steven C. Hageman 1994 - 2004
// This program and source code is free when used with the 
// construction article published in QST 2004. For any other
// use or commercial use you must contact me for a license.
// Fair enough? Thanks - Steve Hageman
//
//-------------------------------------------------------------------
//
// Version Control:
//	B.0 - Beta Release 0 - 6/10/02 - SCH
//  B.1 - Release to support 1st layout - 8/6/02 - SCH
//  B.3 - Final Beta - 16Oct02 - SCH
//  1.0 - First release - 29Dec02 - SCH
//  1.1 - Added power up frequency memory - 13Dec03 
//
//-------------------------------------------------------------------


//-----< Includes >--------------------------------------------------
#include <16F876.h>			// It takes a big PIC to hold all this text
#device *=16				// Enable 16 bit pointers (must always follow device.h)


//-----< PCM Use directives >----------------------------------------
#use delay(clock=10000000)
#fuses HS, NOWDT, PUT, NOPROTECT, NOLVP


//-----< Port definitions >------------------------------------------
#byte	port_a 	= 0x05
#byte	port_b 	= 0x06
#byte	port_c	= 0x07


//-----< Named pin defines >-----------------------------------------
#bit A_SW 			= port_b.6	// Encoder A switch input         
#bit B_SW     		= port_b.7	// Encoder B switch input
#bit DEBUG_MODE		= port_b.0	// If low then PLL debug mode is on

// VFD specific pin functions
#bit DISP_OFF_L		= port_b.5	// If low then VFD is turned off

//-----< RF Deck Pins >----------------------------------------------
#bit DIN 			= port_a.2	// PLL Chip Pins
#bit CLK 			= port_a.0
#bit EN 			= port_a.1
#bit HIGH_BAND_L	= port_b.4  // Band switch relay output


//-----< Constants >-------------------------------------------------
#define FOREVER			1
#define DEBUG_MODE_ON	0		// Debug mode setting
#define TIMEOUT			2400	// 2400 * 0.25 seconds = 
								// 600 seconds or 10 minutes

//-----< Program global variables >----------------------------------
signed int Band;
int Changed;

// Encoder memory
int Last_State;
int New_State;

// Display memory
char Line_1[17];
char Line_2[17];


//-----< LCD/VFD Routines >----------------------------------------------

// Display is connected to port b, definitions below

// Diaplay Connections - 
//  RC5  RC4  RC3  RC2  RC1  RC0 - PORT PINS
//   E   R/S   D7   D6   D5   D4 - LCD/VFD PINS


//-------------------------------------------------------------------
void blip_e_delay()
{
	// Set enable high for 1 cycle then low
	// Designed for 10 MHz clock. Minimum E cycle time is 400 nS.
	#asm
		bsf port_c,5
		nop
		nop
		nop
		bcf port_c,5
	#endasm
	
	// Wait for longest lcd routine to finish, takes time
	// but saves code in not checking for the busy flag.
	delay_ms(5);
}

//-------------------------------------------------------------------
void blip_e()
{
	// Set enable high for 1 cycle then low
	// Designed for 10 MHz clock. Minimum E cycle time is 400 nS.
	#asm
		bsf port_c,5
		nop
		nop
		nop
		bcf port_c,5
	#endasm
	
	// Wait for shortest lcd routine to finish, takes time
	// but saves code in not checking for the busy flag.
	delay_ms(1);
}

//-----< General display routines >------------------------------------

// Decide which type of display to initilize based on 
// the display option pin input
void init_disp(void)
{
	// Clear and initilize LCD/VFD panel to 4 bit mode
	// two line mode, cursor off, display on.
	// RS needs to be low for all of these instructions.
	// -> Takes 60 mS to complete <-
		
	// Send 0x03 three times
	port_c = 0x03;
	blip_e_delay();
	
	port_c = 0x03;
	blip_e_delay();

	port_c = 0x03;
	blip_e_delay();

	// Send 0x02
	port_c = 0x02;
	blip_e_delay();
	
	// send 0x28
	port_c = 0x02;
	blip_e_delay();
	
	port_c = 0x08;
	blip_e_delay();
	
	// send 0x01
	port_c = 0x00;
	blip_e_delay();
	
	port_c = 0x01;
	blip_e_delay();

	// send 0x06
	port_c = 0x00;
	blip_e_delay();
	
	port_c = 0x06;
	blip_e_delay();

	// send 0x0c
	port_c = 0x00;
	blip_e_delay();
	
	port_c = 0x0c;
	blip_e_delay();

	// Send 0x0f to set cursor on with blink
	// Send 0x0e to set cursor on w/o blink

	// Clean up port on exit
	port_c = 0x00;
}

void on_off_vfd(int state)
{
	// Sets the diaplay on or off - Also controls cursor / blinking 
	// Which are set in the second byte to Cursor OFF, Blinking OFF

	// State = 1 is ON, 0 is OFF
	// RS = 0
	port_c = 0x00;
	blip_e();

	port_c = 0x08 | ((state &0x01) << 2);
	blip_e();
}

// Write a byte to the display
void write_disp(byte ch)
{
	// writes the char ch to the lcd. HB then LB.
	// The OR sets RS high 
	port_c = (ch/0x10) | 0x10;          
	blip_e();
	port_c = (ch & 0x0f) | 0x10;
	blip_e();
	
	// Clean up port on exit
	port_c = 0x00;
}

// Set the first line
void first_line()
{
	// Set the LCD to the first display line
	// Send 0x80 to move to the first line.
	port_c = 0x08;
	blip_e();
	
	port_c = 0x00;
	blip_e();
}

// Set the second line
void second_line()
{
	// Set the LCD to the second display line
	// Send 0xc0 for second line.
	port_c = 0x0c;
	blip_e();
	
	port_c = 0x00;
	blip_e();
}


//-----< PLL Routines >----------------------------------------------
	
void setup_pll(void)
{

// PLL setup routine, only needs to be done once, at startup.
// Note: Until control register is set, the PIC clock will be running at
// 1.25 MHz.

	//----- Initilize PLL -----
	// EN High, Din Low, 4 clocks
	CLK = 0;
	DIN = 0;
	EN = 1;

	CLK = 1;
	CLK = 0;	

	CLK = 1;
	CLK = 0;	

	CLK = 1;
	CLK = 0;	

	CLK = 1;
	CLK = 0;	

	// EN Low, Din Low (except clock 4 = 1), 5 clocks
	DIN = 0;
	EN = 0;

	CLK = 1;
	CLK = 0;	

	CLK = 1;
	CLK = 0;	

	CLK = 1;
	CLK = 0;	

	DIN = 1;
	CLK = 1;
	CLK = 0;
	DIN = 0;	

	CLK = 1;
	CLK = 0;	

	EN = 1;
	// PLL is now reset
	
	//----- Setup C Reg with proper bits -----
	EN = 0;
	
	// Bit 7 - PD Polarity, one is inverted phase
	DIN = 1;
	CLK = 1;
	CLK = 0;
	
	// Bit 6 - PD Output
	DIN = 1;	// One is Phase Detector A
	CLK = 1;
	CLK = 0;
	
	// Bit 5 - Lock Detect Off
	DIN = 0;
	CLK = 1;
	CLK = 0;
	
	// Bit 4, 3, 2 - Ref divider output (001 = /1)
	DIN = 0;
	CLK = 1;
	CLK = 0;
	
	DIN = 0;
	CLK = 1;
	CLK = 0;
	
	DIN = 1;
	CLK = 1;
	CLK = 0;

	// Bit 1 - FV off -- NOTE: Set this bit high to get
	// the N counter out on pin 10.
	if (DEBUG_MODE == DEBUG_MODE_ON)
	{
		DIN = 1;
	}
	else
	{
		DIN = 0;
	}
	CLK = 1;
	CLK = 0;
	
	// Bit 0 - FR off -- NOTE: Set this bit high to get
	// the PD ref frequency on pin 9.
	if (DEBUG_MODE == DEBUG_MODE_ON)
	{
		DIN = 1;
	}
	else
	{
		DIN = 0;
	}
	CLK = 1;
	CLK = 0;
	
	DIN = 0;
	
	// EN back high
	EN = 1;

	// Note: The PIC clock will now be 10 MHz
	
	//----- Setup Ref Divider -----
	// 10 MHz / 50 = 200 KHz (50 = 0x0032)
	EN = 0;

	// BIT 14-12 = 000
	DIN = 0;
	CLK = 1;
	CLK = 0;
	
	CLK = 1;
	CLK = 0;
	
	CLK = 1;
	CLK = 0;
	
	// BIT 11-8 = 0000 	
	CLK = 1;
	CLK = 0;
	
	CLK = 1;
	CLK = 0;
	
	CLK = 1;
	CLK = 0;
	
	CLK = 1;
	CLK = 0;
	
	// BIT 7-4 = 0011
	CLK = 1;
	CLK = 0;
	
	CLK = 1;
	CLK = 0;
	
	DIN = 1;
	CLK = 1;
	CLK = 0;
	
	CLK = 1;
	CLK = 0;
	
	// BIT 4-0 = 0010
	DIN = 0;
	CLK = 1;
	CLK = 0;
	
	CLK = 1;
	CLK = 0;
	
	DIN = 1;
	CLK = 1;
	CLK = 0;
	
	DIN = 0;
	CLK = 1;
	CLK = 0;
	
	// Enable Back High
	EN = 1;
}	


void tune_pll(void)
{
int ctr;
unsigned long n_reg;

	//... Calculate the N counter value and set the Band switch

	// Convert Band number to N counter value
	if ( Band <= 24 )
	{
		// Low band
		n_reg = (long)Band*10 + 252;
		HIGH_BAND_L = 1;
	}
	else
	{
		// High band 
		n_reg = (long)Band*5 + 126;
		HIGH_BAND_L = 0;
	}


	//... Now program the PLL

	// Set clock, enable low
	CLK = 0;
	EN = 0;
	
	// Shift in channel frequency to N reg (MSB first)
	for( ctr = 0 ; ctr <= 15 ; ctr++)
	{
		// Decompose the PLL N value into bits, set data pin
		DIN = shift_left(&n_reg, 2, 0);
		
		// Cycle the clock
		CLK = 1;
		CLK = 0;	
	}
	
	// Clean up and set enable high which 
	// transfers count to N reg
	DIN = 0;
	EN = 1;
}


//-----< Message buffers >-------------------------------------------
// These are broken into 4 segments so that 2 can fit in a page of RAM

void msg_1(int pos)
{
	switch(pos)
	{
		case 0:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:1.5MHz 160m");
			strcpy(Line_2, "P.SEL:   1.5/2.0");
			break;
		}
		
		case 1:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:2.0MHz 120m");
			strcpy(Line_2, "P.SEL:   1.5/4.5");

			break;
		}
		
		case 2:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:2.5MHz     ");
			strcpy(Line_2, "P.SEL:   1.5/6.0");
			break;
		}
		
		case 3:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:3.0MHz  90m");
			strcpy(Line_2, "P.SEL:   3.5/1.5");
			break;
		}
		
		case 4:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:3.5MHz  80m");
			strcpy(Line_2, "P.SEL:   3.5/3.0");
			break;
		}
		
		case 5:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:4.0MHz  75m");
			strcpy(Line_2, "P.SEL:   3.5/4.5");
			break;
		}
		
		case 6:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:4.5MHz  60m");
			strcpy(Line_2, "P.SEL:   3.5/5.5");
			break;
		}
		
		case 7:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:5.0MHz     ");
			strcpy(Line_2, "P.SEL:   3.5/5.5");
			break;
		}
		
		case 8:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:5.5MHz     ");
			strcpy(Line_2, "P.SEL:   3.5/6.5");
			break;
		}

		case 9:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:6.0MHz  49m");
			strcpy(Line_2, "P.SEL:   7.0/4.5");
			break;
		}
		
		case 10:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:6.5MHz     ");
			strcpy(Line_2, "P.SEL:   7.0/5.0");
			break;
		}
		
		case 11:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:7.0MHz  41m");
			strcpy(Line_2, "P.SEL:   7.0/5.5");
			break;
		}
		
		case 12:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:7.5MHz     ");
			strcpy(Line_2, "P.SEL:   7.0/6.0");
			break;
		}
		case 13:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:8.0MHz     ");
			strcpy(Line_2, "P.SEL:   7.0/6.5");
			break;
		}
	}
}


void msg_2(int pos)
{
	switch(pos)
	{
		case 14:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:8.5MHz     ");
			strcpy(Line_2, "P.SEL:   7.0/7.0");
			break;
		}
		
		case 15:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:9.0MHz     ");
			strcpy(Line_2, "P.SEL:   7.0/7.0");
			break;
		}
		
		case 16:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:9.5MHz  31m");
			strcpy(Line_2, "P.SEL:   7.0/7.5");
			break;
		}
		
		case 17:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:10.0MHz 30m");
			strcpy(Line_2, "P.SEL:  14.0/5.5");
			break;
		}
		
		case 18:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:10.5MHz    ");
			strcpy(Line_2, "P.SEL:  14.0/5.5");
			break;
		}
		
		case 19:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:11.0MHz    ");
			strcpy(Line_2, "P.SEL:  14.0/6.0");
			break;
		}
		
		case 20:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:11.5MHz 25m");
			strcpy(Line_2, "P.SEL:  14.0/6.0");
			break;
		}
		
		case 21:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:12.0MHz    ");
			strcpy(Line_2, "P.SEL:  14.0/6.5");
			break;
		}
		
		case 22:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:12.5MHz    ");
			strcpy(Line_2, "P.SEL:  14.0/6.5");
			break;
		}
		
		case 23:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:13.0MHz    ");
			strcpy(Line_2, "P.SEL:  14.0/7.0");
			break;
		}
		
		case 24:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:13.5MHz 21m");
			strcpy(Line_2, "P.SEL:  14.0/7.5");
			break;
		}
		
		// Band break here!

		case 25:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:14.0MHz 20m");
			strcpy(Line_2, "P.SEL:  14.0/7.5");
			break;
		}
		
		case 26:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:14.5MHz    ");
			strcpy(Line_2, "P.SEL:  14.0/8.0");
			break;
		}
		
		case 27:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:15.0MHz 19m");
			strcpy(Line_2, "P.SEL:  14.0/8.0");
			break;
		}
		
		case 28:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:15.5MHz    ");
			strcpy(Line_2, "P.SEL:  14.0/8.5");
			break;
		}	
	}
}

void msg_3(int pos)
{
	switch(pos)
	{
		case 29:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:16.0MHz    ");
			strcpy(Line_2, "P.SEL:  21.0/7.0");
			break;
		}
		
		case 30:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:16.5MHz    ");
			strcpy(Line_2, "P.SEL:  21.0/7.0");
			break;
		}
		
		case 31:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:17.0MHz    ");
			strcpy(Line_2, "P.SEL:  21.0/7.5");
			break;
		}
		
		case 32:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:17.5MHz 16m");
			strcpy(Line_2, "P.SEL:  21.0/7.5");
			break;
		}
		
		case 33:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:18.0MHz    ");
			strcpy(Line_2, "P.SEL:  21.0/7.5");
			break;
		}
		
		case 34:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:18.5MHz    ");
			strcpy(Line_2, "P.SEL:  21.0/8.0");
			break;
		}
		
		case 35:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:19.0MHz    ");
			strcpy(Line_2, "P.SEL:  21.0/8.0");
			break;
		}
		
		case 36:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:19.5MHz    ");
			strcpy(Line_2, "P.SEL:  21.0/8.0");
			break;
		}
		
		case 37:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:20.0MHz    ");
			strcpy(Line_2, "P.SEL:  21.0/8.5");
			break;
		}
		
		case 38:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:20.5MHz    ");
			strcpy(Line_2, "P.SEL:  21.0/8.5");
			break;
		}
		
		case 39:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:21.0MHz 15m");
			strcpy(Line_2, "P.SEL:  21.0/9.0");
			break;
		}
		
		case 40:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:21.5MHz 13m");
			strcpy(Line_2, "P.SEL:  21.0/9.0");
			break;
		}
		
		case 41:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:22.0MHz    ");
			strcpy(Line_2, "P.SEL:  21.0/9.5");
			break;
		}
		
		case 42:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:22.5MHz    ");
			strcpy(Line_2, "P.SEL:  21.0/9.5");
			break;
		}
		
		case 43:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:23.0MHz    ");
			strcpy(Line_2, "P.SEL:  28.5/8.0");
			break;
		}	
	}
}


void msg_4(int pos)
{
	switch(pos)
	{
		case 44:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:23.5MHz    ");
			strcpy(Line_2, "P.SEL:  28.5/8.0");
			break;
		}
		
		case 45:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:24.0MHz    ");
			strcpy(Line_2, "P.SEL:  28.5/8.0");
			break;
		}
		
		case 46:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:24.5MHz 12m");
			strcpy(Line_2, "P.SEL:  28.5/8.0");
			break;
		}
		
		case 47:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:25.0MHz    ");
			strcpy(Line_2, "P.SEL:  28.5/8.0");
			break;
		}
		
		case 48:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:25.5MHz 11m");
			strcpy(Line_2, "P.SEL:  28.5/8.5");
			break;
		}
		
		case 49:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:26.0MHz    ");
			strcpy(Line_2, "P.SEL:  28.5/8.5");
			break;
		}
		
		case 50:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:26.5MHz    ");
			strcpy(Line_2, "P.SEL:  28.5/9.0");
			break;
		}
		
		case 51:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:27.0MHz  CB");
			strcpy(Line_2, "P.SEL:  28.5/9.0");
			break;
		}
		
		case 52:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:27.5MHz    ");
			strcpy(Line_2, "P.SEL:  28.5/9.0");
			break;
		}
		
		case 53:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:28.0MHz 10m");
			strcpy(Line_2, "P.SEL:  28.5/9.0");
			break;
		}
		
		case 54:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:28.5MHz 10m");
			strcpy(Line_2, "P.SEL:  28.5/9.5");
			break;
		}
		
		case 55:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:29.0MHz 10m");
			strcpy(Line_2, "P.SEL:  28.5/9.5");
			break;
		}
		
		case 56:
		{				  //0123456789abcdef
			strcpy(Line_1, "FREQ:29.5MHz    ");
			strcpy(Line_2, "P.SEL: 28.5/10.0");
			break;
		}
	}
}


//-----< Display update routine >------------------------------------
void update_disp(void)
{

	// Find the proper message
	if ( Band >= 0 && Band <= 13 )
	{
		msg_1(Band);
	}
	else if ( Band >= 14 && Band <= 28 )
	{
		msg_2(Band);
	}
	else if ( Band >= 29 && Band <= 43 )
	{
		msg_3(Band);
	}
	else if ( Band >= 44 && Band <= 56 )
	{
		msg_4(Band);
	}
	else	// This should never happen...
	{				  //0123456789abcdef
		strcpy(Line_1, "Error Sorting   ");
		strcpy(Line_2, "                ");
	}


	// Update the display now
	first_line();
	printf(write_disp, Line_1);
	second_line();
	printf(write_disp, Line_2);
	
}


//-----< Encoder interrupt >-----------------------------------------
#INT_RB
void encoder(void)
{	
	// Find the new state
	New_State = (A_SW << 1) + B_SW;
	
	// Compare old and new states looking for CW rotation
	if ( Last_State == 0 && New_State == 2 )
	{
		Band += 1;
	}

	else if ( Last_State == 2 && New_State == 3 )
	{
		Band += 1;
	}

	else if ( Last_State == 3 && New_State == 1 )
	{
		Band += 1;
	}

	else if ( Last_State == 1 && New_State == 0 )
	{
		Band += 1;
	}
else
	// If we are here then the rotation was CCW
	Band -= 1;


	// Bound the Band value, preventing rollover
	if ( Band > 56 )
		Band = 56;

	if ( Band < 0 )
		Band = 0;

	// Set the changed flag
	Changed = TRUE;
	Last_State = New_State;
}


//-----< Main Program >----------------------------------------------
main()
{
int last_disp_state;
long time_count;

	//----- Set up port direction reg's 
	//Bit Nums-> 76543210
	set_tris_a(0b00000000);		// PLL Control
	set_tris_b(0b11100001);		// Encoder, option switches
	set_tris_c(0b00000000);		// Display
	port_c = 0;
	port_b = 0;
	port_b_pullups(TRUE);

	//----- Wait for powerup, Initilize LCD 
	delay_ms(12);		// Clock is initially 10MHz / 8 until PLL chip is setup
						// So this delay is really 12 * 8 = 96 mSec
	
	//----- Initilize variables 
	Band = 0;
	last_disp_state = 1;
	Last_State = (A_SW << 1) + B_SW;
	Changed = FALSE;
	time_count = 0;

	//----- Reset PLL, setup C, R regs
	setup_pll();			// Clock is also now 10 MHz
		
	//----- Setup PLL for initial turn on frequency 
	tune_pll();

	//----- Initilize Display
	init_disp();

	//----- Write a startup message
			  //0123456789abcdef
	write_disp("Hagtronics Drake");
	second_line();
	write_disp("Xtal  Eliminator");
	delay_ms(2000);

	// Let one interrupt go by - Clears the encoder
	disable_interrupts(GLOBAL);
	enable_interrupts(INT_RB);
	enable_interrupts(GLOBAL);
	delay_ms(50);
	disable_interrupts(GLOBAL);

	// Read the saved band info
	Band = read_eeprom(0);

	// Make sure the band is valid
	if ( (Band > 56) || (Band < 0) )
	{
		// Saved band must have been invalid, so start over
		Band = 0;
		write_eeprom(0, Band);
	}

	// Clean up display
	init_disp();

	// Set the synthesizer to the last saved value
	update_disp();
	tune_pll();
	
	// Setup encoder interrupt
	disable_interrupts(GLOBAL);
	enable_interrupts(INT_RB);

	//----- Start of main loop		
	while(FOREVER)
	{
		// Disable int, check changed status, update display if needed
		disable_interrupts(GLOBAL);
		if (Changed == TRUE)
		{	
			//Update display
			update_disp();
			tune_pll();
			Changed = FALSE;
			time_count = 0;
		}

		// See if display state changed (only for VFD displays)
		if (last_disp_state != DISP_OFF_L)
		{
			// Check to see if display is ON or OFF
			if ( DISP_OFF_L == 0 )
			{
				// The display is a VFD and it is turned off
				// So, turn it off
				on_off_vfd(0);
			}
			else
			{
				// VFD Display, so make sure it is on
				on_off_vfd(1);
			}

			// Update the last state flag
			last_disp_state = DISP_OFF_L;
		}

		// Check to see if the ten minute wait is over
		if ( time_count > TIMEOUT )
		{
			// Yes ten minutes since last frequency change
			// So check to see if the frequency is different
			if ( Band != read_eeprom(0) )
			{
				// The band is different, so save the new band
				write_eeprom(0, Band);
			}

			// Reset the time counter
			time_count = 0;
		}

		// Increment the time counter
		time_count++;

		// Enable interrupts, hang out for a while...
		enable_interrupts(GLOBAL);
		delay_ms(250);
	
	} // End of main loop
}

//-----< Fini >------------------------------------------------------
